package org.msh.tb.bd.tbforms.dhis2.tb10.blockbuilders;

import org.msh.tb.bd.tbforms.dhis2.TBFormBlockBuilder;
import org.msh.tb.bd.tbforms.dhis2.data.DataValuePayload;
import org.msh.tb.bd.tbforms.query.tb10.TBForm10Block5AQuery;
import org.msh.tb.entities.enums.*;

/**
 * Created by Mauricio on 17/02/2017.
 * Query database and mount TB Form 10 Block 5A Payload to be sent to DHIS2 APIs
 */
public class TBForm10Block5ABuilder extends TBFormBlockBuilder {

    private TBForm10Block5AQuery formQuery;

    @Override
    protected void loadAllFieldsAsZero(DataValuePayload payload) {
        for (DataElementTB10Block5A de : DataElementTB10Block5A.values()) {
            for (COCTypeGenderBlock5A coc : COCTypeGenderBlock5A.values()) {
                payload.mergeDataValue(de.dhis2Id, coc.dhis2CategoryOptionComboId, new Long(0));
            }
        }
    }

    @Override
    protected TBForm10Block5AQuery getBlockQuery() {
        if (formQuery == null) {
            formQuery = new TBForm10Block5AQuery();
        }

        return formQuery;
    }

    @Override
    protected void mountResultPayload(Object[] result, DataValuePayload payload) {
        InfectionSite infectionSite = (InfectionSite) result[0];
        PatientType patientType = (PatientType) result[1];
        CaseDefinition caseDefinition = (CaseDefinition) result[2];
        Gender gender = (Gender) result[3];
        HIVResult hiv = (HIVResult) result[4];
        Long count = (Long) result[5];

        // find dhis2 dataelement id
        String dataElementId = DataElementTB10Block5A.findId(infectionSite, caseDefinition, patientType);

        // merge payload result
        payload.mergeDataValue(dataElementId, COCTypeGenderBlock5A.findId(gender, "ANY"), count);
        // if positive should count in both types (any or positive)
        if (HIVResult.POSITIVE.equals(hiv)) {
            payload.mergeDataValue(dataElementId, COCTypeGenderBlock5A.findId(gender, "POSITIVE"), count);
        }
    }

    /**
     * Catalog that holds the ids of DHIS2 data element used by TB 10 Form Block 1
     * Check in: [DHIS2URL]/api/dataSets/MARm4SH7K0w?fields=id,displayName,dataElements[id,code,name],indicators[id,name]
     */
    public enum DataElementTB10Block5A {
        NEW_UNKNOWN_BAC_PULMONARY("iiNGrYv2MeB", InfectionSite.PULMONARY, CaseDefinition.BACTERIOLOGICALLY_CONFIRMED,  new PatientType[]{ PatientType.NEW, PatientType.UNKNOWN_PREVIOUS_TB_TREAT }),
        NEW_UNKNOWN_CLI_PULMONARY("nLIuyeGvRc4", InfectionSite.PULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED,  new PatientType[]{ PatientType.NEW, PatientType.UNKNOWN_PREVIOUS_TB_TREAT }),
        NEW_UNKNOWN_CLI_EXTRA("zhV41aAMMAz", InfectionSite.EXTRAPULMONARY, CaseDefinition.CLINICALLY_DIAGNOSED,  new PatientType[]{ PatientType.NEW, PatientType.UNKNOWN_PREVIOUS_TB_TREAT }),
        RE_TREATMENT("fEYnYiPMh40", null, null,  new PatientType[]{ PatientType.PREVIOUSLY_TREATED });

        DataElementTB10Block5A(String dhis2Id, InfectionSite infectionSite, CaseDefinition caseDefinition, PatientType[] patientTypes) {
            this.dhis2Id = dhis2Id;
            this.infectionSite = infectionSite;
            this.caseDefinition = caseDefinition;
            this.patientTypes = patientTypes;
        }

        /**
         * The data element id in DHIS2
         */
        private String dhis2Id;

        /**
         * The data element infectionSite
         */
        private InfectionSite infectionSite;

        /**
         * The data element caseDefinition
         */
        private CaseDefinition caseDefinition;

        /**
         * The data element patient types
         */
        private PatientType[] patientTypes;

        /**
         * Returns the DHIS2 dataElement id that matches with the params.
         *
         * @param infectionSite
         * @param caseDefinition
         * @param patientType
         * @return
         */
        public static String findId(InfectionSite infectionSite, CaseDefinition caseDefinition, PatientType patientType) {
            // find the id
            if (infectionSite == null || caseDefinition == null || patientType == null) {
                throw new RuntimeException("Infection site, case definition and patient type must not be null");
            }

            for (DataElementTB10Block5A elem : values()) {
                if((elem.infectionSite == null || elem.infectionSite.equals(infectionSite))
                        && (elem.caseDefinition == null || elem.caseDefinition.equals(caseDefinition))) {

                    for (PatientType pt : elem.patientTypes) {
                        if (pt.equals(patientType)) {
                            return elem.dhis2Id;
                        }
                    }

                }
            }

            throw new RuntimeException("DataElementTB10Block5: Data element not found");
        }

        public String getDhis2Id() {
            return dhis2Id;
        }
    }

    /**
     * Catalog that holds the ids of DHIS2 categories option combos used by TB 10 Form Block 5
     * COC = Category Option Combo (DHIS2 data delimiter)
     */
    public enum COCTypeGenderBlock5A {

        POSITIVE_FEMALE("hvRREr8GjzN", "POSITIVE", Gender.FEMALE),
        POSITIVE_MALE("GofBVjc5qrY", "POSITIVE", Gender.MALE),
        TESTED_MALE("Rpymec1hNoR", "ANY", Gender.MALE),
        TESTED_FEMALE("JfpmE6R3vyZ", "ANY", Gender.FEMALE);

        COCTypeGenderBlock5A(String dhis2CategoryOptionComboId, String type, Gender gender) {
            this.dhis2CategoryOptionComboId = dhis2CategoryOptionComboId;
            this.type = type;
            this.gender = gender;
        }

        /**
         * The category option combo id in DHIS2
         */
        private String dhis2CategoryOptionComboId;

        /**
         * The category option combo type (examinated or positive)
         */
        private String type;

        /**
         * The category option combo gender
         */
        private Gender gender;


        /**
         * Returns the DHIS2 categoryOptionCombo id that matches with the gender and type on params.
         * @param gender
         * @param type
         * @return
         */
        public static String findId(Gender gender, String type){
            if (gender == null || type == null) {
                throw new RuntimeException("Gender and type must not be null");
            }

            for (COCTypeGenderBlock5A cat : values()) {
                if (gender.equals(cat.gender) && type.equals(cat.type)) {
                    return cat.dhis2CategoryOptionComboId;
                }
            }

            throw new RuntimeException("COCTypeGenderBlock5: Category not found");
        }
    }
}
